#! /usr/local/bin/R
# ------------------------------------------------------------------------------------------


# ------------------------------------------------------------------------------------------
message("Log file for code executed at\n")
message(format(Sys.time(), "%a %b %d %X %Y"))
# ------------------------------------------------------------------------------------------


# ------------------------------------------------------------------------------------------
library(devtools)

library(ggplot2)
library(ggsci)
library(grid)
library(scales)
library(gridExtra)
library(patchwork)
library(tikzDevice)
library(bit64);
library(magrittr);
library(readxl)
library(stringr);
library(glue)
library(haven)
library(zeallot)

library(data.table);
library(statar)
# ------------------------------------------------------------------------------------------


# ------------------------------------------------------------------------------------------
latex_percent <- function (x) {
  x <- plyr::round_any(x, scales:::precision(x)/100)
  stringr::str_c(comma(x * 100), "\\%")
}
# ------------------------------------------------------------------------------------------


# ------------------------------------------------------------------------------------------
# var = "log_import"
prep_plot_DiD <- function(var, controls="gravity", eia ="FTA"){
  dt1 <- fread(glue("./output/coef_for_plots/coef_for_plot_postest_{controls}_{eia}_{var}.txt"), skip=3)
  setnames(dt1, c("lag", "coef", "se", "pval"))
  dt1[, coef := as.numeric(coef) ]
  dt1 <- dt1[ !is.na(coef) & !(lag %in% c("N", "r2") ) ]
  dt1[, horizon := as.numeric(gsub("\\D", "", lag)) ]
  dt1 <- dt1[!is.na(horizon), .(year_diff=fifelse(str_sub(lag, 1, 1)=="l", -1, 1) * horizon, 
                                coef=as.numeric(coef), se=as.numeric(se) ) ]
  return(dt1)
}

# --- plots 
make_plot_DiD <- function(dt_plot_var, xlab="", ylab="", se_mult=2.0, horizon=10){
  p1 <- dt_plot_var[ abs(year_diff) <= horizon ] %>%
    ggplot(aes(x=year_diff, y=coef)) + 
    geom_line(alpha=0.55) + geom_point(shape=1, alpha = 1.0, colour="black") + 
    geom_errorbar(aes(ymin=coef-se_mult*se, ymax=coef+se_mult*se), 
                  colour="black", alpha=0.5, width=0.15) +
    geom_vline(xintercept=0, color="red", linetype="dashed") + 
    geom_hline(yintercept=0, color="black", linetype="solid") + 
    theme_bw() + 
    theme(axis.text = element_text(size=8), ) + 
    scale_y_continuous(labels = latex_percent) + 
    # scale_y_continuous(labels = scales::percent) + 
    xlab(xlab) + ylab(ylab) +
    theme(axis.title=element_text(size=9))
  return(p1)    
}
# ------------------------------------------------------------------------------------------


# ------------------------------------------------------------------------------------------
# build a large dataset of all results
dlist = c("PTA", "FTA", "CUMECU", "PTA_un", "FTA_un", "CUMECU_un", "PTA_mult", "FTA_mult", "CUMECU_mult")
dt_full_plots = data.table()

for (var in dlist){
  for (lhs in c("win_log_import", "R2_y", "fx_vol", "var_adj", "beta")){
    dt_tmp = prep_plot_DiD(lhs, eia = var)
    dt_tmp[, `:=`(lhs=lhs, deal_type=var)]
    dt_tmp = rbind(dt_tmp, cbind(data.table(year_diff=0, coef=0, se=NA), dt_tmp[, .(lhs, deal_type)]))
    dt_full_plots = rbind(dt_full_plots, dt_tmp)
  }
}    
dt_full_plots = unique(dt_full_plots)
dt_full_plots[, deal_mult := fcase(grepl("_un", deal_type), "bilateral", 
                                   grepl("_mult", deal_type), "multilateral", default="all"), 
              by=seq(1,nrow(dt_full_plots)) ]
dt_full_plots[, deal_type := gsub("_mult", "", gsub("_un", "", deal_type) ) ]

# # not bilateral
# p1 = dt_full_plots[ year_diff<=10 & year_diff>=-5 ] %>% 
#   ggplot(aes(x=year_diff, y=coef, colour=deal_type)) + 
#   geom_line(alpha=0.55) + geom_point(shape=1, alpha = 1.0) + 
#   geom_errorbar(aes(ymin=coef-2*se, ymax=coef+2*se), 
#                 alpha=0.5, width=0.15) +
#   geom_vline(xintercept=0, color="red", linetype="dashed") + 
#   geom_hline(yintercept=0, color="black", linetype="solid") + 
#   theme_bw() + 
#   theme(axis.text = element_text(size=8), ) + 
#   scale_y_continuous(labels = latex_percent) + 
#   facet_grid(lhs ~ deal_mult, scales="free") + 
#   # scale_y_continuous(labels = scales::percent) + 
#   # xlab(xlab) + ylab(ylab) +
#   theme(axis.title=element_text(size=9))
# # ggsave("./output/figures/event_combined_all.pdf", p1, width=16, height=14)

for (lhs_iter in c("win_log_import", "R2_y", "fx_vol", "var_adj", "beta")){
  dt_tmp = dt_full_plots[ lhs == lhs_iter & year_diff<=10 & year_diff>=-10 ][ order(deal_mult, year_diff) ]
  
  plot_deal_type = function(df){
    df %>% 
      ggplot(aes(x=year_diff, y=coef, colour=deal_type)) + 
      geom_line(alpha=0.55) + geom_point(shape=1, alpha = 1.0) + 
      geom_errorbar(aes(ymin=coef-2*se, ymax=coef+2*se), 
                    alpha=0.5, width=0.15) +
      geom_vline(xintercept=0, color="red", linetype="dashed") + 
      geom_hline(yintercept=0, color="black", linetype="solid") + 
      theme_bw() + theme(axis.text = element_text(size=8), ) + 
      scale_y_continuous(labels = latex_percent) +
      xlab(glue("Years from Trade Deal, {df$deal_mult[1]} Deals")) + ylab("")
  }
  assign(glue("p_{lhs_iter}_mult"), plot_deal_type(
    dt_full_plots[ lhs == lhs_iter & year_diff<=10 & year_diff>=-10 & deal_mult=="multilateral"][ 
      order(deal_mult, year_diff) ]
  ) )
  assign(glue("p_{lhs_iter}_bi"), plot_deal_type(
    dt_full_plots[ lhs == lhs_iter & year_diff<=10 & year_diff>=-10 & deal_mult=="bilateral"][ 
      order(deal_mult, year_diff) ]
  ) )
  assign(glue("p_{lhs_iter}_all"), plot_deal_type(
    dt_full_plots[ lhs == lhs_iter & year_diff<=10 & year_diff>=-10 & deal_mult=="all"][ 
      order(deal_mult, year_diff) ]
  ) )
}

p_log_import = (p_win_log_import_mult  + theme(legend.position=c(0.8, 0.15), legend.title=element_blank()) + 
                  coord_cartesian(ylim=c(-0.25, 0.5)) ) + 
  (p_win_log_import_bi + coord_cartesian(ylim=c(-0.25, 0.5)) + theme(legend.position="none") )
# p_log_import

p_fx_vol = (p_fx_vol_mult + 
              theme(legend.position=c(0.35, 0.15), legend.title=element_blank(), 
                    legend.key.size = unit(0.5, 'cm'), legend.text = element_text(size=6)) +
              coord_cartesian(ylim=c(-0.035, 0.02)) + ggtitle("Volatility, $\\sigma$") ) + 
  (p_fx_vol_bi + ylab("") + coord_cartesian(ylim=c(-0.035, 0.02)) + theme(legend.position="none") )
# p_fx_vol
p_R2 = (p_R2_y_mult + theme(legend.position="none") + 
          theme(legend.position=c(0.35, 0.15), legend.title=element_blank(), 
                legend.key.size = unit(0.5, 'cm'), legend.text = element_text(size=6)) +
          coord_cartesian(ylim=c(-0.35, 0.25)) + ggtitle("R-squared, $R^2$") ) + 
  (p_R2_y_bi + ylab("")+ coord_cartesian(ylim=c(-0.35, 0.25)) + theme(legend.position="none") )
# p_R2
p_var_adj = (p_var_adj_mult + theme(legend.position="none") + 
               theme(legend.position=c(0.35, 0.15), legend.title=element_blank(), 
                     legend.key.size = unit(0.5, 'cm'), legend.text = element_text(size=6)) +
               coord_cartesian(ylim=c(-0.35, 0.25)) + ggtitle("Unshared risk, $\\rho$") ) + 
  (p_var_adj_bi + ylab("") + coord_cartesian(ylim=c(-0.35, 0.25)) + theme(legend.position="none") )
p_beta = (p_beta_mult + theme(legend.position="none") + 
            theme(legend.position=c(0.35, 0.15), legend.title=element_blank(), 
                  legend.key.size = unit(0.5, 'cm'), legend.text = element_text(size=6)) +
            coord_cartesian(ylim=c(-0.51, 0.4)) + ggtitle("Base Loading, $\\beta$") ) + 
  (p_beta_bi + coord_cartesian(ylim=c(-0.51, 0.41)) + theme(legend.position="none") ) 
# p_beta

p_all = p_fx_vol / p_var_adj / p_R2 / p_beta;

tikz("./output/figures/figuresA1_A2.tex", width=7, height=9.5)
p_all;
dev.off()

ggsave("./output/figures/figuresA1_A2.pdf", p_all, width=12, height=16)
# --- postscript for publication 
# ggsave(filename = "./output/figures/figuresA1_A2.eps", 
#        plot = p_all,
#        width=12, height=16, units="in",
#        device = cairo_ps)
# ggsave(filename = "./output/figures/figuresA1_A2.ps",
#        plot = p_all,
#        width=12, height=16, units="in",
#        device = cairo_ps)

# ---
# no deal separation main table:
p_log_import = (p_win_log_import_all  + theme(legend.position=c(0.8, 0.15), legend.title=element_blank()) + 
                  coord_cartesian(ylim=c(-0.25, 0.5)) )
p_fx_vol = (p_fx_vol_all + 
              theme(legend.position=c(0.35, 0.15), legend.title=element_blank(), 
                    legend.key.size = unit(0.5, 'cm'), legend.text = element_text(size=6)) +
              coord_cartesian(ylim=c(-0.035, 0.02)) + ggtitle("Volatility, $\\sigma$") ) 
p_R2 = (p_R2_y_all + theme(legend.position="none") + 
          theme(legend.position=c(0.35, 0.15), legend.title=element_blank(), 
                legend.key.size = unit(0.5, 'cm'), legend.text = element_text(size=6)) +
          coord_cartesian(ylim=c(-0.2, 0.2)) + ggtitle("R-squared, $R^2$") ) 
p_var_adj = (p_var_adj_all + theme(legend.position="none") +  
               theme(legend.position=c(0.35, 0.15), legend.title=element_blank(), 
                     legend.key.size = unit(0.5, 'cm'), legend.text = element_text(size=6)) +
               coord_cartesian(ylim=c(-0.25, 0.25)) + ggtitle("Unshared risk, $\\rho$") ) 
p_beta = (p_beta_all + theme(legend.position="none") + 
            theme(legend.position=c(0.35, 0.15), legend.title=element_blank(), 
                  legend.key.size = unit(0.5, 'cm'), legend.text = element_text(size=6)) +
            coord_cartesian(ylim=c(-0.51, 0.4)) + ggtitle("Base Loading, $\\beta$") ) 
p_all = p_fx_vol / p_var_adj / p_R2 / p_beta;

ggsave("./output/figures/figure3.pdf", p_all, width=8, height=12)
tikz("./output/figures/figure3.tex", width=8, height=12)
  p_all;
dev.off()

# --- postscript for publication 
# ggsave(filename = "./output/figures/figure3.eps", 
#        plot = p_all,
#        width=12, height=16, units="in",
#        device = cairo_ps)
# ggsave(filename = "./output/figures/figure3.ps",
#        plot = p_all,
#        width=12, height=16, units="in",
#        device = cairo_ps)
# ------------------------------------------------------------------------------------------

